import java.io.*;
import java.util.Hashtable;
import java.util.Vector;
import java.applet.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.rmi.*;
import java.rmi.registry.*;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.AccessControlException;
import dSelfComp;
import dSelf.*;

/**
 * The class dSelfVM represents the virtual machine of dSelf and is
 * the root of the whole system. All its contents are static, because
 * there is only one dSelfVM running all the time and its public 
 * variables like its name or the lobby object must be easy accessible
 * to all other objects.
 */
public class dSelfVM{
  
  /** The version of this dSelf VM*/
  public static final String version = "dSelf - version 1.1.2";
  /** The actual directory path, where the files are located */
  protected static String dirPath = "";
  /** The chosen front-end of the VM. Either the terminal or GUI */
  protected static VMFrontEnd frontEnd = null;
  /** The name of this VM, that will be bound to the registry */
  public    static String VMName = "dSelfVM";
  /** The server, that offers the access for other VMs */
  public    static Server server = null;
  /** A list of all known remote VMs */
  public    static RemoteVMs remoteVMs = new RemoteVMs();
  /** Should the name of this VM be bound or rebound to the registry ?*/
  protected static boolean rebindName = false;
  /** Should the VM start in verbose mode ? */
  public static boolean verboseMode = false;
  /** Should the registry be created by the VM or not ?*/
  protected static boolean createRegistry = false;
  /** The lobby object, which must be accessible to all other objects */
  public static    LocalOrdinarySO lobbySO = new LocalOrdinarySO("lobby");
  /** The file that is started at the begin */
  protected static Vector filenames = new Vector();
  /** The ever grow uniq gene id */
  protected static long geneID = 1;
  
 /**
  * The starting point of the VM. 
  *
  * @param args The arguments given by the user 
  */         
  public static void main(String[] args){

    // Check the arguments 
    checkArgs(args);
    
    // Tell the Java VM to run the finalize methods at the end. The
    // remote objects need this to unbind their names at the rmiregistry.
    /*System.runFinalizersOnExit(true);*/
    
    // When no security manager exists then we install one
    if(System.getSecurityManager() == null)
      System.setSecurityManager(new RMISecurityManager());

    // Check if the registry is already running. If not create a new one.
    boolean regIsRunning = true;
    
    if(createRegistry){ 
      try{
        Naming.list("");
      }catch(ConnectException e){
        regIsRunning = false;
      }catch(MalformedURLException e){
        System.err.println("\nRMI-Error: "+e);
        System.exit(1);
      }catch(RemoteException e){
        System.err.println("\nRMI-Error: " + e);
        System.exit(1);
      }
    }
    
    // Start the server to provide remote access for other VMs. If the 
    // RMI-Registry isn't already running we'll start a new one.
    try{
      if(!regIsRunning && createRegistry){
	if(verboseMode)
	  System.out.print("Created a new registry.................................");

        LocateRegistry.createRegistry(1099);  
	
	if(verboseMode && createRegistry)
          System.out.println("OK");    
      }	

      server = new Server();

      // Bind or rebind this VM to the registry. 
      if(rebindName){
        if(verboseMode)
          System.out.print("Try to rebind the name of the VM to the RMI-Registry...");
        Naming.rebind (VMName, server);
      }else{
        if(verboseMode)
          System.out.print("Try to bind the name of the VM to the RMI-Registry.....");
        Naming.bind (VMName, server);
      }	
      if(verboseMode)
        System.out.println("OK");
    }
    catch(AlreadyBoundException e){
      System.err.println(
       "\nRMI-Error: the name \""+VMName+"\" is already bound to the RMI-Registry !\n"+
       "Possibly there is another VM currently running with this name or a \n"+
       "previous VM couldn't unbind it (e.g. it has been killed). In the second\n"+
       "case you can try \"-r "+VMName+"\" to rebind the name.");

      System.exit(1);
    }catch(ConnectException e){
      System.err.println(
       "\nRMI-Error:" + e + "\nHave you started the RMI-Registry ?" +
       "\nIf not, you can start it manual with \"rmiregistry\" or let it"+
       "\nstart with the argument \"-cr\" by the dSelfVM. In the second"+
       "\ncase make sure, that no other VMs on the local host will run,"+
       "\nbecause when this dSelf-VM is stopped the registry will also be stopped !");
      System.exit(1);
    }catch(RemoteException e){
      System.err.println("\nRMI-Error: " + e);
      System.exit(1);
    }catch (AccessControlException e) {
      System.err.println ("\nAccessControlError: " + e +
                          "\nHave you set the policy-file correctly ?");
      System.exit(1);
    }catch (Exception e) {
      System.err.println ("\nError: " + e );
      System.exit(1);
    }     
    
    // The dSelf-world with the basic objects will be generated
    if(verboseMode)
      System.out.print("Initialize the dSelf-world.............................");    

    initializeSelfWorld();

    if(verboseMode)
      System.out.println("OK");    
       
    // If frontEnd wasn't set to the terminal front-end at checkArgs 
    // the prefered front-end is a JPanel in a JFrame
    if(frontEnd == null){
      if(verboseMode)
        System.out.print("Generate the GUI.......................................");    

      frontEnd = new JPanelFrontEnd();

      JPanelFrontEnd.frame = new JFrame("SelfVM - "+VMName);
      JPanelFrontEnd.frame.addWindowListener(new WindowAdapter() {
        public void windowClosing(WindowEvent e) {System.exit(0);}
      });
      JPanelFrontEnd.frame.getContentPane().add("Center", (JPanelFrontEnd)frontEnd);
      JPanelFrontEnd.frame.pack();
      JPanelFrontEnd.frame.setVisible(true);  

      if(verboseMode)
        System.out.println("OK");    
    }
    
    // Force the compile to send its error messages to the VM.
    dSelfComp.calledByVM = true;

    frontEnd.Run(filenames);
  }
    
 /**
  * Exits the VM. All destructors of all remote objects are
  * called in order to delete all their remote references and
  * the name of this VM is unbinded at the registry. 
  */
  public static void exit(){

    lobbySO = null;
    System.runFinalization();
    
    try{  
      Naming.unbind(VMName);
    }catch(Exception e){
      System.err.println("RMI-Error: " + e);
      System.exit(1);
    }
    
    System.exit(0);  
  }
  
 /**
  * When this is called, the initial dSelf-world beginning from
  * the lobby will be created.
  */
  protected static void initializeSelfWorld(){ 

    lobbySO.addSlot(new ParentSlot("systemObjects", new LocalOrdinarySO("systemObjects")));      
    lobbySO.addSlot(new ParentSlot("vm", new LocalOrdinarySO("vm")));      
    
    LocalOrdinarySO systemObjects = (LocalOrdinarySO)lobbySO.getSlotContent("systemObjects");
 
    systemObjects.addSlot(new DataSlotImpl("nil", new NilSO()));      
    systemObjects.addSlot(new DataSlotImpl("true", new TrueSO()));      
    systemObjects.addSlot(new DataSlotImpl("false", new FalseSO()));      
    systemObjects.addSlot(new DataSlotImpl("vector", new LocalObjectVectorSO()));      
    systemObjects.addSlot(new DataSlotImpl("byteVector", new ByteVectorSO()));      
 
    LocalOrdinarySO vm = (LocalOrdinarySO)lobbySO.getSlotContent("vm");

    vm.addSlot(new DataSlotImpl("local", lobbySO));      
  }
  
 /**
  * Calls the compiler with the given string as argument and 
  * evaluates its resulting expression. The return value is the
  * name of the resulting object, e.g. "nil", "an object", etc.
  *
  * @param input The input of the user at the shell
  * @return The output of the evaluated expression
  */
  public static String evalInput(String input){
  
    try{
      CodeExpr expr = dSelfComp.parseString(input); 
      DataSO dSO = null;

      // When an error occured at parse time, then the resulting
      // expression won't be evaluated    
      if(!Globals.errorOccured){
        // When a null-pointer is returned, then the input was an
	// empty expression and nothing will be done.
        try{ 
          dSO = (expr != null) ? expr.eval(lobbySO) : null;
        }catch(NonLocalReturnException e){
	  //System.err.println("lobby catch NonLRE");
	  dSO = e.getReturnValue();
        }
        return dSO == null ? null : dSO.getName();
      }
    }catch(dSelfException e){
      printRuntimeError(e.getMessage());
    }catch(StackOverflowError e){
      printRuntimeError("Stack overflow !" +
	  "Have you checked the termination of all loops/recursions ?");
    }

    return "";
  }
  
 /**
  * Load a script with the given name cancatenated with the actual 
  * dirpath from file.
  *
  * @param fileName the name of the script
  * @return An expression, that represents the compiled script
  */
  public static CodeExpr loadScriptDirPath(String fileName) 
         throws dSelfException{
  
    if(fileName.charAt(0) == File.separatorChar)
      return loadScript(fileName);
    else
      return loadScript(dirPath + fileName);
  }
  
 /**
  * Loads a script with the given name from file. When a precompiled
  * class-file with the same name and newer date exists, it will be 
  * linked to running environment. Otherwise it loads the script and
  * forwards its content to the compiler.
  *
  * @param fileName the name of the script
  * @return An expression, that represents the compiled script
  */
  public static CodeExpr loadScript(String fileName) 
         throws dSelfException{

    if(fileName.endsWith(dSelfComp.dSelfSuffix))
      fileName = fileName.substring(0, fileName.length() - dSelfComp.dSelfSuffix.length());

    String className = null;
    File classFile = null;
    
    try{
      classFile = new File(fileName + ".class");
      className = classFile.getName();
      
      if(classFile.exists() && classFile.isFile()){
        File dSelfFile = new File(fileName + dSelfComp.dSelfSuffix);
      
        if(dSelfFile.exists()){
	  if(dSelfFile.lastModified() < classFile.lastModified()){
	    printMessage("Loading precompiled file \""+ fileName + ".class\" ..."); 
	    return loadFromClass(className.substring(0,className.length()-6));		   
	  }else{
	    printWarning("\"" + fileName + ".class\" is out"+
	         " of date, load \"" + fileName + dSelfComp.dSelfSuffix + "\" instead.");
	  }   
	}  
      }
    }catch(SecurityException e){
      printWarning("No permission to examine \"" + fileName +".class\" !" + e +
            "\nLoad \"" + fileName + dSelfComp.dSelfSuffix + "\" instead.");
    }catch(InstantiationException e){
      printWarning("Cannot instantiate \"" + fileName +".class\" !\n" + e +
            "\nLoad \"" + fileName + dSelfComp.dSelfSuffix + "\" instead.");
    }catch(IllegalAccessException e){
      printWarning("Cannot access \"" + fileName +".class\" !\n" + e +
            "\nLoad \"" + fileName + dSelfComp.dSelfSuffix + "\" instead.");
    }catch(ClassNotFoundException e){
      // System.setProperty("java.class.path", ...) don't changes
      // the CLASSPATH of the class loader, so print a warning to the
      // user. 
      String neededClassPath = classFile.getAbsolutePath().substring(0, 
	  classFile.getAbsolutePath().length() - className.length());

      printWarning("Cannot find \"" + fileName +".class\" !\n" + e +
          "\nAre you sure that \"" + neededClassPath +"\" is in your CLASSPATH ?" +
          "\nLoad \"" + fileName + dSelfComp.dSelfSuffix + "\" instead.");
    }catch(NoClassDefFoundError e){
      printWarning("Cannot load \"" + fileName +".class\" !\n" + e +
          "\nLoad \"" + fileName + dSelfComp.dSelfSuffix + "\" instead.");
    } 
      
    return dSelfComp.parseFile(fileName + dSelfComp.dSelfSuffix);
  }
  
 /**
  * Loads a script by linking its corresponding class-file to
  * the running environment.
  *
  * @param fileName the name of the precompiled script
  * @return An expression, that represents the compiled script
  */
  public static CodeExpr loadFromClass(String fileName)
      throws InstantiationException, IllegalAccessException,
	     ClassNotFoundException{

    Class cl = Class.forName(fileName);

    return ((dSelfProgram) cl.newInstance()).getExpression();
  }

 /**
  * Sets a new value for the dirpath
  *
  * @param dir The old dirpath
  * @return The previous value of the dirpath
  */
  public static String setDirPath(String dir){
  
    String oldPath = dirPath;
  
    dirPath = dir + File.separator;
  
    return oldPath;
  }

 /**
  * Returns the actual value of the dirpath
  */
  public static String getDirPath(){
  
    return dirPath;
  }  
  
 /**
  * Prints an error message, that was detected at compile time, on the 
  * chosen front-end of the VM.
  * 
  * @param errMsg The Message, that describes this kind of error
  */  
  public static void printCompiletimeError(String errMsg){
  
    frontEnd.printMessage("Compiler error: " + errMsg);
    Globals.errorOccured = true; 		     
  }

 /**
  * Prints an error message, that was detected at compile time, on the 
  * chosen front-end of the VM with details of its occurrence.
  *
  * @param errMsg The Message, that describes this kind of error
  * @param line The line, where the error occured
  * @param col The column, where the error occured
  */  
  public static void printCompiletimeError(String errMsg, int line, int col){
  
    frontEnd.printMessage("Compiler error at line "+(line+1)+", column "+(col+1)+" :\n"+
                     "-- "+errMsg);
    Globals.errorOccured = true; 		     
  }

 /**
  * Prints an error message, that was detected at runtime, on the 
  * chosen front-end of the VM.
  * 
  * @param errMsg The Message, that describes this kind of error
  */  
  public static void printRuntimeError(String errMsg){
  
    frontEnd.printMessage("Runtime error: " + errMsg);
  }

 /**
  * Prints a message on the chosen front-end of the VM.
  *
  * @param msg The message, that should be printed
  */  
  public static void printMessage(String msg){
  
    frontEnd.printMessage(msg);
  }
     
 /**
  * Prints a warning message on the chosen front-end of the VM.
  *
  * @param msg The warning message, that should be printed
  */  
  public static void printWarning(String msg){

    frontEnd.printMessage("Warning: " + msg);
  }
  
 /**
  * Prints the usage of the VM on the standard output stream.
  */     
  protected static void printUsage(){
  
    System.out.println("Usage:  dSelfVM [options]"); 
    System.out.println(""); 
    System.out.println("Possible options are:\n"); 
    System.out.println("-h        give this help"); 
    System.out.println("-v        verbose, give informations when starting"); 
    System.out.println("-t        start with terminal front-end"); 
    System.out.println("-cr       create registry when not found"); 
    System.out.println("-f <name> load script when starting"); 
    System.out.println("-b <name> bind the VM to the given name (default: dSelfVM)"); 
    System.out.println("-r <name> rebind the VM to the given name"); 
    System.out.println("-ds       activate debugger of scanner at start"); 
    System.out.println("-dc       activate parser-debugger(CUP) at start"); 
    System.out.println("-dp       activate parse-tree debugger (with parenthesis)"); 
    System.out.println("-di       activate parse-tree debugger (with indentation)"); 
    System.out.println("-dl       activate search path debugger (lookup algorithm)"); 
    System.out.println("-dC       activate lookup cache debugger (lookup algorithm)"); 
  }
  
 /**
  * Parses the arguments, which were given by the user and sets 
  * corresponding environment variables.
  */ 
  protected static void checkArgs(String[] args){

    for(int i=0; i<args.length; i++){
      
      if(args[i].equals("-b")){
        // "-b" binds the name of this VM to the registry  
        i++;
	if(i<args.length){
	  VMName = args[i];
	  rebindName = false;
	}else{
	  System.err.println("Missing argument after -b !");
	  System.exit(1);
	}     
      }else if(args[i].equals("-r")){
        // "-r" rebinds the name of this VM to the registry  
        i++;
	if(i<args.length){
	  VMName = args[i];
	  rebindName = true;
	}else{
	  System.err.println("Missing argument after -r !");
	  System.exit(1);
	}     
      }else if(args[i].equals("-f")){
        // "-f" loads the file with this name
        i++;
	if(i<args.length){
	  filenames.add(args[i]);
	}else{
	  System.err.println("Missing argument after -f !");
	  System.exit(1);
	}     
      }else if(args[i].equals("-h")){
        // "-h" prints the usage of the VM	
        printUsage();
	System.exit(0);
      }else if(args[i].equals("-t")){
        // "-t" starts the VM with the terminal front-end 	
        frontEnd = new TerminalFrontEnd();
      }else if(args[i].equals("-dc")){
        // "-dc" enables the debugger of CUP
        Globals.debug_cup = true; 
      }else if(args[i].equals("-dp")){ 
        // "-dp" enables the debugger, that creates flat parse-trees
        Globals.debug_parsetree_flat = true; 
	Globals.debug = true;   
      }else if(args[i].equals("-di")){
        // "-di" enables the debugger, that creates indented parse-trees
        Globals.debug_parsetree_indent = true; 
        Globals.debug = true;           
      }else if(args[i].equals("-ds")){
        // "-ds" enables the debugger of the scanner
        Globals.debug_scanner = true; 
      }else if(args[i].equals("-dl")){
        // "-dl" enables the debugger, that displays the search-trees
	// of the lookup alogorithm
	Globals.debug_searchPath = true; 
      }else if(args[i].equals("-dC")){
        // "-dC" enables the debugger, that displays the lookup cache info
	Globals.debug_lookupCache = true; 
      }else if(args[i].equals("-v")){
        // "-v" starts in verbose mode
	verboseMode = true; 
      }else if(args[i].equals("-cr")){
        // "-cr" creates the registry by the VM.
	createRegistry = true; 
      }else{
        System.err.println("Unknown parameter: "+args[i]);
	System.exit(1);
      }
    }
   
    // Display the chosen settings
    if(verboseMode){
      System.out.println("Name of VM..................................\"" 
         +VMName+"\" "+(rebindName ? "(rebound)" : "(bound)"));
      System.out.println("Chosen front-end............................"+ 
          (frontEnd instanceof TerminalFrontEnd ? "Terminal" : "GUI"));
      System.out.println("Load file..................................."+ 
          (filenames.size() == 0 ? "none" : "\"" + filenames + "\""));
      System.out.println("debugger of scanner........................."+
          (Globals.debug_scanner ? "enabled" : "disabled"));
      System.out.println("debugger of parser(CUP)....................."+
          (Globals.debug_cup ? "enabled" : "disabled")); 	  	  
      System.out.println("Generate flat parse-trees..................."+
          (Globals.debug_parsetree_flat ? "enabled" : "disabled")); 	  	  
      System.out.println("Generate indented parse-trees..............."+  
          (Globals.debug_parsetree_indent ? "enabled" : "disabled")); 	  	  
      System.out.println("Show search path of lookup algorithm........"+
          (Globals.debug_searchPath ? "enabled" : "disabled")); 
      System.out.println("Show lookup cache info of lookup algorithm.."+
          (Globals.debug_lookupCache ? "enabled" : "disabled")); 
      System.out.println();	  	  	  
    }
  }   

  public static Long getNextGeneCode() {

    if (geneID == Long.MAX_VALUE)
      throw new RuntimeException("All gene IDs are exhausted, restarting application is recommended.");

    if (Globals.debug_lookupCache)
      frontEnd.printMessage("Gene ID " + geneID + "	allocated");
    
    return new Long(geneID++);
  }
}

