import java.io.*;
import java_cup.runtime.Symbol;
import parser.*;
import dSelf.*;

/**
 * The class dSelfComp represents the compiler of dSelf. It consists
 * of a scanner, generated by JFlex and a parser, generated by CUP.
 */ 
public class dSelfComp{

  /** The name of the Java compiler */
  protected static String javac = "javac";
  /** The classpath for the Java compiler */
  protected static String classpath = ".";
  /** Should the generated Java class be compiled by the Java compiler ?*/
  protected static boolean javaCompile = false;
  /** The prefered size (number of characters) for the generated Java methods*/
  protected static int javaMethodSize = 10000;
  /** Was the compiler started by the VM or by the user as a standalone application ?*/
  public static boolean calledByVM;

  /** dSelf source file suffix */
  public static String dSelfSuffix = ".dsf";

 /**
  * This method makes the compiler accessible to the user. 
  * 
  * @param args The parameters of the compiler
  */       
  public static void main(String[] args){
    
    calledByVM = false;
    Globals.genJava = true;
    
    // Check the arguments and compile the specified script 
    try{
      parseFile(checkArgs(args));
    }catch(dSelfException e){
      reportError(e.getMessage());  
    }  
  }

 /**
  * Parse a given string and return its corresponding expression
  *
  * @param parseStr The string, that consists of dSelf-code
  * @return An expression, that represents the compiled code
  */  
  public static CodeExpr parseString(String parseStr)
      throws dSelfException{
  
    Result res = parse(new dSelfParser(new dSelfScanner(new StringReader(parseStr)))); 
     
    return res == null ? null : 
        (res.code instanceof CodeExpr ? (CodeExpr) res.code : null);
  }

 /**
  * Parse a dSelf-script with the given namen. If {@link dSelf.Globals#genJava}
  * is set, then a Java class will be generated.
  *
  * @param fileName The name of the script
  * @return An expression, that represents the compiled code
  */  
  public static CodeExpr parseFile(String fileName) 
      throws dSelfException{

    Result res = null;
    
    isCorrectName(fileName);  

    try{
      res = parse(new dSelfParser(new dSelfScanner(new FileReader(fileName))));  
    }catch(FileNotFoundException e){
      throw new dSelfException("File not found: " + fileName, "", "ioError");
    }
    
    if(Globals.errorOccured)
      return null;
      
    if(Globals.genJava)
      System.out.println("-- compile "+fileName);
    
    if(res != null && Globals.genJava)
      genJavaClass(fileName, res.javaCode);         
 
    return res.code instanceof CodeExpr ? (CodeExpr) res.code : null;
  }

 /**
  * Starts the parser and gives debug information, when selected. 
  *
  * @param parser The parser, that should be used
  * @return The {@link parser.Result} of the parser
  * @see dSelf.Globals Globals for details about the debugging options
  */   
  protected static Result parse(dSelfParser parser) throws dSelfException{
  
    // Den Fehleranzeiger zurcksetzen
    Globals.errorOccured = false;
       
    // Eingabe parsieren und eventuell Debuginformationen ausgeben  
    try{
      Symbol CUP_Result = Globals.debug_cup ? 
	  parser.debug_parse() : parser.parse();
		     			     
      Result res = CUP_Result != null ? 
          ((Result)(CUP_Result).value) : null;  

      if(res != null){ 
        if(Globals.debug_parsetree_flat) 
          System.out.println(res.tree+'\n');         
      
        if(Globals.debug_parsetree_indent) 
          printParseTree(res.tree);         

        return res;
      }
    }catch(dSelfException e){ 
      throw e;
    }catch(Exception e){ 
      System.err.println("Internal error: ");
      e.printStackTrace();
    }
    
    return null;
  } 
 
 /**
  * Check if a given name is a correct name for a Java class. If not,
  * we can't create a Java class.
  *
  * @param name The name of the class
  */
  private static void isCorrectName(String name) 
      throws dSelfException{
  
    boolean nameOK = true;
    int pos = 0;
    
    name = (new File(name)).getName();

    if(name.endsWith(dSelfComp.dSelfSuffix))
      name = name.substring(0, name.length()-dSelfComp.dSelfSuffix.length());
      
    if(Character.isJavaIdentifierStart(name.charAt(0))){
      for(pos=1; pos<name.length(); pos++){
        if(!Character.isJavaIdentifierPart(name.charAt(pos))){
	  nameOK = false;
	  break;
	}
      }
    }else
      nameOK = false;
        
    if(!nameOK)
      throw new dSelfException("Name must be a correct Java-identifier ! "+
              "Character '"+name.charAt(pos)+"' is not allowed !", "",
	      "primitiveFailedError");
  }
  
 /**
  * Generates a Java class with the given name. When {@link dSelfComp#javaCompile}
  * is set, then the java compiler is called with the given
  * {@link dSelfComp#classpath}. The size of the generated method
  * is specified in {@link dSelfComp#javaMethodSize}.
  *
  * @param name The name for the Java class
  * @param codeTree The tree with the expressions, generated by the parser
  */ 
  protected static void genJavaClass(String name, JavaCodeTree codeTree)
      throws dSelfException{

    try{
      String filename = name.endsWith(dSelfComp.dSelfSuffix) ? 
              name = name.substring(0, name.length()
				    - dSelfComp.dSelfSuffix.length()) : name;

      FileOutputStream file = new FileOutputStream(filename + ".java");

      file.write((codeTree.getJavaClass(new File(filename).getName(), 
          javaMethodSize)).getBytes());

      file.close();

      System.out.println();
       
      if(!javaCompile)
        return;
	
      System.out.println("-- compile "+filename+".java");

      // Start the external Java compiler. 
      Runtime run = Runtime.getRuntime();
      Process proc = run.exec(javac+" -classpath "+classpath
                            +" "+filename+".java");

      try{
        // Warning ! It can happen, that the process never stops. 
        // The reason then is probably platform specific, e.g.
        // when the linux shell don't returns the output of the
        // compiler, because the error message is too long when the
	// method size is about 60 kBytes. This can result in a
	// deadlock. Because of this, the Java compiler option 
	// isn't activated by default.
        proc.waitFor();
	
	byte[] bytes = null;
	int readBytes = 0;
	String result = "";
	InputStream stream = null;
	  
	stream = proc.getInputStream();
	if(stream.available() != 0){
  	  bytes = new byte[stream.available()];
	  readBytes = stream.read(bytes);
	  result += new String(bytes, 0, readBytes-1);
        }
	  
        result += "\n";

	stream = proc.getErrorStream();
	if(stream.available() != 0){
  	  bytes = new byte[stream.available()];
	  readBytes = stream.read(bytes);
	  result += new String(bytes, 0, readBytes-1);
        }

	System.out.println(result);
      }catch(InterruptedException e){
        System.out.println("Error: The compile process has been interrupted !");
      } 	  
    }catch(IOException e){
      throw new dSelfException("I/O-Error: "+e.getMessage(), "", "ioError");
    }      
    
  }
  
 /**
  * Prints an error message on the error stream, when the compiler
  * is running as a standalone application. When it was called by
  * the VM, the message is forwarded to {@link dSelfVM#printCompiletimeError}.
  *
  * @param msg The message that descibes the error
  */ 
  public static void reportError(String msg){

    Globals.errorOccured = true;  

    if(calledByVM)
      dSelfVM.printCompiletimeError(msg);
    else
      System.err.println("Error: " + msg);  
  }

 /**
  * Prints an error message on the error stream, when the compiler
  * is running as a standalone application. When it was called by
  * the VM, the message is forwarded to {@link dSelfVM#printCompiletimeError}.
  *
  * @param msg The message that descibes the error
  * @param line The line, where the error occured
  * @param col The column, where the error occured
  */ 
  public static void reportError(String msg, int line, int col){
  
    Globals.errorOccured = true;  

    if(calledByVM)
      dSelfVM.printCompiletimeError(msg, line, col);
    else
      System.err.println("Error at line "+(line+1)+", column "+(col+1)+" :\n"+
                     "-- " + msg);
  }
      
 /**
  * Parses the arguments, which were given by the user and sets 
  * corresponding environment variables.
  */ 
  protected static String checkArgs(String[] args){
  
    String filename = "";
     
    // Wenn Argumente fehlen nur "Usage" ausgeben 	 		        
    if(args.length == 0){
      printUsage();
      System.exit(0);
    }
   
    for(int i=0; i<args.length; i++){

      if(args[i].equals("-h")){
        printUsage();
	System.exit(0);
      }else if(args[i].equals("-r")){	
        Globals.genJava = false; 
      }else if(args[i].equals("-j")){	
        javaCompile = true;
      }else if(args[i].equals("-jm")){	
        i++;
        javaMethodSize = Integer.parseInt(args[i]);
	if(javaMethodSize < 100 || javaMethodSize > 65000){
	  System.err.println("Wrong value for method size ! Expected was \n"+
	        "a number between 100 and 65000 and not "+javaMethodSize+".");
	  System.exit(0);
	}  
      }else if(args[i].equals("-jc")){
        i++;
	javac = args[i];	
      }else if(args[i].equals("-c")){	
        i++;
	classpath = args[i]; 
      }else if(args[i].equals("-dc")){	
        Globals.debug_cup = true;
      }else if(args[i].equals("-dp")){	
        Globals.debug_parsetree_flat = true; 
	Globals.debug = true;
      }else if(args[i].equals("-di")){	
        Globals.debug_parsetree_indent = true; 
	Globals.debug = true;
      }else if(args[i].equals("-ds")){	
        Globals.debug_scanner = true;
      }else{	
        filename = args[i];
      }
    }
    
    if(filename == ""){
      System.err.println("Missing name of file");
      System.exit(1);
    }      
    
    return filename;
  }
  
 /**
  * Prints the usage of the VM on the standard output stream.
  */     
  protected static void printUsage(){
  
    System.out.println("Usage:  dSelfComp [options] <file>"); 
    System.out.println(""); 
    System.out.println("Possible options are:\n"); 
    System.out.println("-h         give this help"); 
    System.out.println("-r         recognizer (don't generate code)"); 
    System.out.println("-j         invoke java-compiler and generate class file"); 
    System.out.println("-jc <name> the name of the java-compiler (default: javac)"); 
    System.out.println("-jm <size> set prefered size for methods(default: 10000 Bytes)"); 
    System.out.println("-c  <path> set the CLASSPATH for java compiler (default: \".\")"); 
    System.out.println("-ds        print debug-informations of scanner"); 
    System.out.println("-dc        print debug-informations of parser (CUP)"); 
    System.out.println("-dp        print the parse-tree (with parenthesis)"); 
    System.out.println("-di        print the parse-tree (with indentation)"); 
    
  }
  
 /**
  * Prints the parse-tree with indentations.
  *
  * @param flat_parse_tree The parse-tree in a flat form with parenthesis
  */
  protected static void printParseTree(String flat_parse_tree){
 
    int depth=0;
    StringBuffer tree = new StringBuffer();
    
    for(int i=0; i<flat_parse_tree.length(); i++){

      switch(flat_parse_tree.charAt(i)){
      
        case '(': depth++;
	          tree.append("\n");   
	          for(int j=0; j<depth; j++)
		    tree.append("-");
		  break;
	
        case ')': depth--;
		  break;
            
        case ',': tree.append("\n");   
	          for(int j=0; j<depth; j++)
		    tree.append("-");
		  break;
		  
		      	
	default: tree.append(flat_parse_tree.charAt(i));
      }       	  
    }
    if (calledByVM)
      dSelfVM.printMessage(tree + "\n");
    else
      System.out.println(tree);
  }
}
