package dSelf;

import java.rmi.*;
import java.rmi.server.*;
import java.util.Vector;
import java.util.Stack;
import java.util.Hashtable;
import java.util.Enumeration;
import dSelfVM;

/**
 * Server makes it possible, that objects of remote dSelf virtual 
 * machines can access objects of this dSelf virtual machine.
 */
public class Server extends UnicastRemoteObject implements ServerInterface {

  /** A vector of all references that target this dSelf-VM. */
  protected Vector references = new Vector();
  
  /** A stack to prevent leaks in the indices of the references vector. */
  protected Stack vectorLeaksStack = new Stack();

 /**
  * Creates a new server.
  */     
  public Server() throws RemoteException{

    references.setSize(1000);
    vectorLeaksStack.setSize(1000);
    for(int i=vectorLeaksStack.size()-1; i>=0; i--)
      vectorLeaksStack.push(new Integer(i));
  }

 /**
  * Creates a new reference to the lobby of this dSelf-VM.
  * 
  * @return The ID of the new reference. 
  */ 
  public int connectToLobby(){

    return addRemoteReference((RemoteReferenceSO)dSelfVM.lobbySO);
  }
  
 /**
  * Creates a new reference to the given object.
  *
  * @param refObj The object, that shall be refered
  * @return The ID of the new reference
  */   
  protected int addRemoteReference(RemoteReferenceSO refObj){
  
    // If the stack is empty, we must reserve new memory for the vector
    // and store its indices into the stack
    if(vectorLeaksStack.empty()){
      int oldSize = references.size();
      
      references.setSize(oldSize+1000);
      for(int i=0; i<1000; i++)
        vectorLeaksStack.set(i, new Integer(i+oldSize));
    }
    
    // Put the object we refer to into the vector at an unused index
    // determined by the stack of index-leaks
    int ID = ((Integer)vectorLeaksStack.pop()).intValue();
    references.set(ID, refObj);
    
    return ID;
  }
  
 /**
  * Removes the reference at the specified index.
  */   
  public void removeRemoteReference(int ID){
  
    references.set(ID, null);
    vectorLeaksStack.push(new Integer(ID));
  }

 /**
  * Returns the object at the specified position.
  *
  * @param ID The ID of the reference
  * @return The object for that ID
  */ 
  protected dSelfObject getObjAt(int ID){
  
    return ((dSelfObject)references.get(ID));
  }
  
 /**
  * The name of the object with this ID.
  *
  * @param ID The ID of the reference
  * @return The name of that object
  */ 
  public String getNameOf(int ID){

    return getObjAt(ID).getName();
  }
  
 /**
  * Returns the content of the slot with the given name of the object 
  * with the specified ID.
  *
  * @param ID The ID of the reference
  * @param slotName The name of the slot
  * @return The content of the demanded slot
  */ 
  public SerializedSO getSlotContentOf(int ID, String slotName){
	
    DataSO dataSO = (DataSO)getObjAt(ID);
    dSelfObject content = dataSO.getSlotContent(slotName);

    // If the found object is an ordinary object, an object vector or
    // an assignment object then we create a reference to it.
    if(content instanceof OrdinarySO)
      return new SerializedOrdinarySO(
        addRemoteReference((RemoteReferenceSO)content));
         
    if(content instanceof ObjectVectorSO)
      return new SerializedObjectVectorSO(
        addRemoteReference((RemoteReferenceSO)content));

    if(content instanceof AssignmentSO)
      return new SerializedAssignmentSO(
        addRemoteReference((RemoteReferenceSO)content));

    if(content instanceof BlockSO)
      return new SerializedBlockSO((BlockSO)content, this);
    
    // All primitive objects except object vectors and blocks are 
    // serializable, so just give them back. 
    if(content instanceof PrimitiveSO)
      return (SerializedSO)content;

    if(content instanceof MethodSO)
      return new SerializedMethodSO((MethodSO)content, this);

    return null;  
  }

 /**
  * Returns the parent objects of the object with this ID.
  *
  * @param ID The ID of the reference
  * @return An arry with all parent objects
  */
  public SerializedSO[] getParentsOf(int ID){
  
    Vector parVec = ((DataSO)getObjAt(ID)).getParentVector();
    SerializedSO parents[] = new SerializedSO[parVec.size()];
    
    for(int i=0; i<parVec.size(); i++){
      if(parVec.get(i) instanceof OrdinarySO)
        parents[i] = new SerializedOrdinarySO(
            addRemoteReference((RemoteReferenceSO)parVec.get(i))); 
      else if(parVec.get(i) instanceof ObjectVectorSO)     
        parents[i] = new SerializedObjectVectorSO(
            addRemoteReference((RemoteReferenceSO)parVec.get(i)));
      else 
        parents[i] = (SerializedSO)parVec.get(i);	     
    }
    
    return parents;
  }
  
 /**
  * Returns the slots of the object with this ID.
  *
  * @param ID The ID of the reference
  * @return A slot vector with all slots.
  */ 
  public SerializedSlotVector getSlotsOf(int ID){
  
    return new SerializedSlotVector(
             ((DataSO)getObjAt(ID)).getSlotVector(), this);
  }
  
 /**
  * Removes all slots of the object with this ID.
  *
  * @param ID The ID of the reference
  */ 
  public void removeAllSlotsOf(int ID){
  
    ((OrdinarySO)getObjAt(ID)).removeAllSlots();
  }

 /**
  * Removes the slot with the specified name of the object with 
  * this ID.
  * 
  * @param ID The ID of the reference
  * @param slotName The name of the slot 
  */  
  public void removeSlotOf(int ID, String slotName) 
      throws dSelfException{
  
    ((OrdinarySO)getObjAt(ID)).removeSlot(slotName);
  }
  
 /**
  * Adds some slots to the object with this ID. Slots with the same
  * name are replaced by the new ones.
  * 
  * @param ID The ID of the reference
  * @param VMName The name of the dSelf-VM, where the caller of this
  * method is located
  * @param remSlots The remote slots, that are added to the object with
  * the specified ID
  */ 
  public void addSlotsTo(int ID, String VMName, SerializedSlotVector remSlots)
         throws dSelfException{

    try{
      ((OrdinarySO)getObjAt(ID)).addSlots(remSlots.getSlotVector(
       dSelfVM.remoteVMs.addServer("//"+getClientHost()+"/"+VMName)));
    }catch(ServerNotActiveException e){
      throw new dSelfException("The Server is not activ !");
    }   
  }

 /**
  * Adds some slots to the object with this ID. Slots with the same
  * name aren't replaced by the new ones.
  * 
  * @param ID The ID of the reference
  * @param VMName The name of the dSelf-VM, where the caller of this
  * method is located
  * @param remSlots The remote slots, that are added to the object with
  * the specified ID
  */ 
  public void addSlotsIfAbsentTo(int ID, String VMName, 
      SerializedSlotVector remSlots) throws dSelfException{

    try{
      ((OrdinarySO)getObjAt(ID)).addSlotsIfAbsent(
           remSlots.getSlotVector(dSelfVM.remoteVMs.addServer(
	   "//"+getClientHost()+"/"+VMName)));
    }catch(ServerNotActiveException e){
      throw new dSelfException("The Server is not activ !");
    }   
  }
  
 /**
  * Replaces the content of the slot with the given slotname, located
  * in the object with this ID, with the new content.
  *
  * @param ID The ID of the reference
  * @param VMName The name of the dSelf-VM, where the caller of this
  * method is located
  * @param obj The new content of the slot
  */ 
  public void setContentOf(int ID, String VMName, SerializedSO obj)
      throws dSelfException{
  
    DataSO content = null;
    
    try{
      // The remote object is an OrdinarySO, so return the reference to it
      if(obj instanceof SerializedOrdinarySO){
        content = new RemoteOrdinarySO(dSelfVM.remoteVMs.addServer("//"+
            getClientHost()+"/"+VMName), ((SerializedOrdinarySO)obj).getID());
      }else if(obj instanceof SerializedObjectVectorSO){
        content = new RemoteObjectVectorSO(dSelfVM.remoteVMs.addServer("//"+
            getClientHost()+"/"+VMName), ((SerializedObjectVectorSO)obj).getID());
      }else if(obj instanceof SerializedBlockSO){
        content = ((SerializedBlockSO)obj).getBlock(dSelfVM.remoteVMs.addServer("//"+
            getClientHost()+"/"+VMName));
      }else{
        content = (DataSO) obj;
      }
    }catch(ServerNotActiveException e){
      throw new dSelfException("The Server is not activ !");
    }   

    ((AssignmentSO)getObjAt(ID)).setContent(content);
  }

 /**
  * Returns the length of the object vector with the specified ID.
  *
  * @param ID The ID of the reference
  */   
  public int getLengthOfObjectVector(int ID){
  
    return ((ObjectVectorSO)getObjAt(ID)).getLength();
  }

 /**
  * Returns the object vector at the specified position.
  *
  * @param ID The ID of the reference
  * @param pos The position in the object vector 
  * @return The object at this position
  */ 
  public SerializedSO getObjectOfObjectVector(int ID, int pos){
  
    ObjectVectorSO objectVector = (ObjectVectorSO)getObjAt(ID);
    DataSO content = objectVector.getObjectAt(pos);

    // If the found object is an ordinary object, then we create
    // a reference to it and give a SerializedOrdinarySO back.
    if(content instanceof OrdinarySO)
      return new SerializedOrdinarySO(
        addRemoteReference((RemoteReferenceSO)content));
         
    // If the found object is an object vector, then we create
    // a reference to it and give a SerializedObjectVectorSO back.
    if(content instanceof ObjectVectorSO)
      return new SerializedObjectVectorSO(
        addRemoteReference((RemoteReferenceSO)content));

    // If it isn't an ordinary object or an object vector, it must be
    // a primitive object that can be serialized.
    return (SerializedSO)content;
  }
  
 /**
  * Puts an object at specified position in the object vector with this
  * ID.
  *
  * @param ID The ID of the reference
  * @param VMName The name of the dSelf-VM, where the caller of this
  * method is located
  * @param pos The position in the object vector 
  * @param so The object, that will be stored in the object vector   
  */ 
  public void putObjectOfObjectVector(int ID, String VMName, int pos, 
      SerializedSO so) throws dSelfException{

    ObjectVectorSO objectVector = (ObjectVectorSO)getObjAt(ID);
    
    try{
      if(so instanceof SerializedOrdinarySO)  
        objectVector.putObjectAt(pos, new RemoteOrdinarySO(
          dSelfVM.remoteVMs.addServer("//"+getClientHost()+"/"+VMName),
	  ((SerializedOrdinarySO)so).getID()));
      else if(so instanceof SerializedObjectVectorSO)  
        objectVector.putObjectAt(pos, new RemoteObjectVectorSO(
          dSelfVM.remoteVMs.addServer("//"+getClientHost()+"/"+VMName),
  	  ((SerializedObjectVectorSO)so).getID()));
      else
        objectVector.putObjectAt(pos, (PrimitiveSO)so); 
    }catch(ServerNotActiveException e){
      throw new dSelfException("The Server is not activ !");
    }   
  }
  
  private static class Tupel{
  
    private int value1, value2;
    
    public Tupel(int val1, int val2){
    
      value1 = val1;
      value2 = val1;
    }
    
    public boolean equals(int val1, int val2){
      
      return (value1 == val1) && (value2 == val2); 
    }  
  }
      
  private Hashtable lockHash = new Hashtable();
  
 /**
  * Locks the object at specified position in the object vector with this
  * ID.
  *
  * @param ID The ID of the reference
  * @param hash The hashcode of the remote object that will lock 
  */
  public void lockObject(int ID, int hash){
  
    // We must create a new local object, that represents the remote
    // object for locking purpose.
    LocalOrdinarySO dummyObj = new LocalOrdinarySO();

    // Store the dummy object, so that we can find it for a later
    // unlock 
    lockHash.put(new Tupel(ID, hash), dummyObj);

    // Now lock the target with the dummy that represents the remote
    // object
    ((DataSO)getObjAt(ID)).lock(dummyObj);
  } 
  
 /**
  * Unlocks the object at specified position in the object vector with this
  * ID.
  *
  * @param ID The ID of the reference
  * @param hash The hashcode of the remote object that will lock 
  */
  public void unlockObject(int ID, int hash) throws dSelfException{
  
    Enumeration keys = lockHash.keys();
    Tupel tup = null;
   
    // Find a matching key  
    while(keys.hasMoreElements()){
      tup = (Tupel)keys.nextElement();
      if(tup.equals(ID, hash))
        break;
    }
    
    ((DataSO)getObjAt(ID)).unlock((DataSO)lockHash.remove(tup));
  }	

 /**
  * Returns the hashcode of the object with the given ID.
  *
  * @param ID The ID of the reference
  */    
  public int getHashCodeOf(int ID){
  
    return ((OrdinarySO)getObjAt(ID)).getHashCode();
  } 

 /**
  * Returns the name of the VM of the object with the given ID.
  *
  * @param ID The ID of the reference
  */    
  public String getLocationOf(int ID){
  
    return ((OrdinarySO)getObjAt(ID)).getLocation();
  }   
}

