package dSelf;

import java.util.Vector;
import dSelfVM;

/**
 * OrdinaryMsgExpr represents the expressions of ordinary messages in 
 * dSelf ({@link OrdinaryMsg}). 
 */
public class OrdinaryMsgExpr extends MessageExpr{

  /** The name of this message. */
  protected String msgName = null;
  /** The name of the receiver, if it is resended. */
  protected String resendTo = null;
  /** Cached lookup result */
  protected PolymorphicInlineCache lookupCache = new PolymorphicInlineCache();
  
 /**
  * Creates a new ordinary message expression, that is send to the receiver,
  * that is given as the first argument. The name of this message is 
  * specified by "name". 
  *
  * @param rec The receiver of the message
  * @param name The selector of this message
  */ 
  public OrdinaryMsgExpr(CodeExpr rec, String name){
  
    receiver = rec;
    msgName = name;
  }

 /**
  * Creates a new ordinary message expression, that is send to the receiver,
  * that is given as the first argument. The name of this message is 
  * specified by "name". It is resent to the object with the name 
  * specified in "res".
  *
  * @param rec The receiver of the message
  * @param name The selector of this message
  * @param res The name of the object, where this message is resent to
  */ 
  public OrdinaryMsgExpr(CodeExpr rec, String name, String res){
  
    receiver = rec;
    msgName = name;
    resendTo = res;
  }

 /**
  * Creates a new ordinary message expression, that is send to the receiver,
  * that is given as the first argument. The name of this message is 
  * specified by "name". It is resent to the object with the name 
  * specified in "res" with one argument.
  *
  * @param rec The receiver of the message
  * @param name The selector of this message
  * @param res The name of the object, where this message is resent to
  * @param arg The argument of the message    
  */ 
  public OrdinaryMsgExpr(CodeExpr rec, String name, String res, CodeExpr arg){
  
    receiver = rec;
    msgName = name;
    resendTo = res;
    methodArguments.add(arg);
  }
  
 /**
  * Creates a new ordinary message expression, that is send to the receiver,
  * that is given as the first argument. The name of this message is 
  * specified by "name". It is resent to the object with the name 
  * specified in "res" with its arguments, that are inside the vector.
  *
  * @param rec The receiver of the message
  * @param name The selector of this message
  * @param res The name of the object, where this message is resent to
  * @param args The arguments of the message    
  */ 
  public OrdinaryMsgExpr(CodeExpr rec, String name, String res, Vector args){
  
    receiver = rec;
    msgName = name;
    resendTo = res;
    methodArguments = args;  
  }
  
 /**
  * Creates a new ordinary message expression, that is send to the receiver,
  * that is given as the first argument. The name of this message is 
  * specified by "name". It has one argument.
  *
  * @param rec The receiver of the message
  * @param name The selector of this message
  * @param arg The argument of the message    
  */ 
  public OrdinaryMsgExpr(CodeExpr rec, String name, CodeExpr arg){
  
    receiver = rec;
    msgName = name;
    methodArguments.add(arg);
  }  
    
 /**
  * Creates a new ordinary message expression, that is send to the receiver,
  * that is given as the first argument. The name of this message is 
  * specified by "name". Its arguments are inside the vector.
  *
  * @param rec The receiver of the message
  * @param name The selector of this message
  * @param args The arguments of the message    
  */ 
  public OrdinaryMsgExpr(CodeExpr rec, String name, Vector args){
  
    receiver = rec;
    msgName = name;
    methodArguments = args;  
  }  
  
 /** 
  * Creates a new ordinary message object, that represents the evaluated 
  * form of this ordinary message expression.
  *
  * @param dSOVec A vector with the arguments of this message
  * @return The new created message
  */
  protected Message createMessage(Vector dSOVec){

    return new OrdinaryMsg(msgName, dSOVec, resendTo);
  }

 /** 
  * Evaluates this message expression within the given context and returns 
  * the result caused by this message. 
  *
  * @param dSo The context, within this message expression is evaluated
  * @return The resulting dSelf object
  */
  public DataSO eval(DataOrMethodSO selfContext) 
    throws dSelfException, NonLocalReturnException{
  
    Vector dSOVec = new Vector();
    DataOrMethodSO rec;
    LookupResult lr;
    Genes genes;
    
    for(int i=0; i<methodArguments.size(); i++)
      dSOVec.add(((CodeExpr)methodArguments.elementAt(i)).eval(selfContext));

    rec = (receiver==null) ? selfContext : receiver.eval(selfContext);
    genes = rec.getSlotVector().getGenes();
    lr = lookupCache.get(genes);

    if (Globals.debug_lookupCache || Globals.debug_searchPath)
      dSelfVM.printMessage("...."+this+" current lookupCache => "+lookupCache);

    if (lr == null) {
      lr = rec.lookupMsg((OrdinaryMsg)createMessage(dSOVec));
      if (Globals.debug_lookupCache || Globals.debug_searchPath)
	dSelfVM.printMessage("....found by dispatch: "+lr);

      if (lr.getCategory() == LookupResult.NORMAL_METHOD) {
	lookupCache.put(genes, lr);

	if (Globals.debug_lookupCache || Globals.debug_searchPath)
	  dSelfVM.printMessage("...."+this+" new lookupCache +> "+lookupCache);
      }
    }else {
      if (Globals.debug_lookupCache || Globals.debug_searchPath)
	dSelfVM.printMessage("....found in OrdinaryMsgExpr PIC: "+lr);
    }

    if(Globals.debug_searchPath)
      dSelfVM.printMessage("...evaluate slot \"" + msgName + "\"");

    if (lr.getCategory() == LookupResult.NORMAL_DATA)
      return rec.evalSO(lr.getHolder().getSlotContent(msgName), dSOVec, msgName);
    
    return rec.evalSO(lr.getValue(), dSOVec, msgName);
  }

}
