package dSelf;
 
import java.io.*;
import java.util.Vector;
import dSelfVM;

/**
 * The class LongSO repesents the long numbers of dSelf.
 * LongSO is a primitive dSelf object and so it has only one
 * parent slot called "parent", that is shared by all longs.
 * The object, that "parent" refers to has initialy only one
 * parent slot, also called "parent" that refers to the lobby.
 * This object is dedicated to include the user-defined methods,
 * (e.g. +, -, * ...) that are shared by all longs.
 */
public class LongSO extends SerializedPrimitiveSO{
 
  /** The value of the long object */
  public long value = 0;

 /** The shared parent object for all longs */
  protected static LocalOrdinarySO parent = 
    new LocalOrdinarySO(new ParentSlot("parent", dSelfVM.lobbySO));

 /**
  * The shared slots for all longs. It consists of only one slot
  * called "parent"
  */
  protected static SlotVector slotVector = 
    new SlotVector(new ParentSlot("parent",parent));   
    
 /**
  * The shared parent slots for all longs. It consists of only one slot
  * called "parent"
  */
  protected static Vector parVector = new Vector();   
  
  static{
    parVector.add(parent);   
  }
  
 /**
  * Creates new dSelf long with the given Java long value
  */  
  public LongSO(long longValue){

    value = longValue;
  }
    
 /**
  * Creates new dSelf long with the given Java Long value
  */  
  public LongSO(Long longValue){

    value = longValue.longValue();
  }

 /**
  * Returns the slots of the longs. I.e, it's only one slot
  * called "parent".
  */   
  public SlotVector getSlotVector(){
    
    return slotVector; 
  }
  
 /**
  * Returns the shared parent slot for all dSelf longs.
  */  
  protected LocalOrdinarySO getParent(){
  
    return parent;
  }
    
 /**
  * Returns the parent slots of the longs. I.e, it's only one slot
  * called "parent".
  */   
  public Vector getParentVector(){
  
    return parVector;
  }   

 /**
  * Returns the value of this long in the form "long(...)".
  */
  public String getName(){
  
    return "long("+value+")";
  }

 /**
  * The dispatcher for dSelf longs, that receives the primitive
  * messages, that are dedicated to it, and executes the actions.
  * 
  * @param msg The primitive message, that shall be executed
  * @return The result of the called primitive message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
         throws dSelfException, NonLocalReturnException{

    try{
    
      switch(msg.getMessageID()){

        case PrimMsg.EQ:                     
	  return _Eq(msg.getFirstArg());  

        case PrimMsg.LONGADD:                     
	  return _LongAdd(msg.getFirstArg());
	
        case PrimMsg.LONGAND:                     
	  return _LongAnd(msg.getFirstArg());
	
        case PrimMsg.LONGARITHMETICSHIFTRIGHT:                     
	  return _LongArithmeticShiftRight(msg.getFirstArg());
	
        case PrimMsg.LONGASDOUBLE:                     
	  return _LongAsDouble();
	
        case PrimMsg.LONGASFLOAT:                     
	  return _LongAsFloat();
	
        case PrimMsg.LONGASINT:                     
	  return _LongAsInt();
	
        case PrimMsg.LONGASSHORT:                     
	  return _LongAsShort();

        case PrimMsg.LONGDIV:                     
	  return _LongDiv(msg.getFirstArg());
	
        case PrimMsg.LONGEQ:                     
	  return _LongEQ(msg.getFirstArg());
	
        case PrimMsg.LONGGE:                     
	  return _LongGE(msg.getFirstArg());
	
        case PrimMsg.LONGGT:                     
	  return _LongGT(msg.getFirstArg());
	
        case PrimMsg.LONGLE:                     
	  return _LongLE(msg.getFirstArg());
	
        case PrimMsg.LONGLT:                     
	  return _LongLT(msg.getFirstArg());
	
        case PrimMsg.LONGLOGICALSHIFTLEFT:                     
	  return _LongLogicalShiftLeft(msg.getFirstArg());
	
        case PrimMsg.LONGLOGICALSHIFTRIGHT:                     
	  return _LongLogicalShiftRight(msg.getFirstArg());
	
        case PrimMsg.LONGMOD:                     
	  return _LongMod(msg.getFirstArg());
	
        case PrimMsg.LONGMUL:                     
	  return _LongMul(msg.getFirstArg());
	
        case PrimMsg.LONGNE:                     
	  return _LongNE(msg.getFirstArg());
	
        case PrimMsg.LONGOR:                     
	  return _LongOr(msg.getFirstArg());
	
        case PrimMsg.LONGSUB:                     
	  return _LongSub(msg.getFirstArg());
	
        case PrimMsg.LONGXOR:                     
	  return _LongXor(msg.getFirstArg());
      }

      return super.dispatchPrimMsg(msg);       
    }
    catch(dSelfException e){
    
      return execFailBlock(msg, e);
    }
  }

 /**
  * Checks, if the argument is equal to the value of this long object.
  *
  * @param arg The argument must be of type LongSO
  */
  protected BooleanSO _Eq(dSelfObject so) throws dSelfException{

    checkIfLongSO("_Eq:", so);
    return BooleanSO.asBooleanSO(((LongSO)so).value == value);
  }  

 /**
  * Adds another dSelf long to this long
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected LongSO _LongAdd(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongAdd:", so);
    return new LongSO(value + ((LongSO)so).value); 
  }  
  
 /**
  * Returns a bitwise AND of this long and the argument
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */ 
  protected LongSO _LongAnd(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongAnd:", so);  
    return new LongSO(value & ((LongSO)so).value); 
  }  
  
 /**
  * Arithmetic right shift of this long by the numbers of bits indicated
  * by the argument. The sign bit is preserved
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */  
  protected LongSO _LongArithmeticShiftRight(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongArithmeticShiftRight:", so);  
    return new LongSO(value >> ((LongSO)so).value);
  }    
  
 /**
  * Converts this dSelf long to a dSelf double.  
  */
  protected DoubleSO _LongAsDouble(){ 
  
    return new DoubleSO(value); 
  }  

 /**
  * Converts this dSelf long to a dSelf float.  
  */
  protected FloatSO _LongAsFloat(){
  
    return new FloatSO((float) value);  
  }  

 /**
  * Converts this dSelf long to a dSelf integer.  
  */
  protected IntegerSO _LongAsInt(){
  
    return new IntegerSO((int) value); 
  }  

 /**
  * Converts this dSelf long to a dSelf short.  
  */
  protected ShortSO _LongAsShort(){
  
    return new ShortSO((short) value); 
  }  

 /**
  * Divides this long through another short
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected LongSO _LongDiv(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongDiv:", so);  
    try{  
      return new LongSO(value / ((LongSO)so).value); 
    }
    catch(ArithmeticException e){
      if(((LongSO)so).value == 0)
        throw new dSelfException(" Division by zero -> "+value+
	    " _LongDiv: 0 !", "_LongDiv:", "divisionByZeroError");
    }  
    
    return null;
  }  
  
 /**
  * Checks, if the argument is equal to the value of this long object.
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected BooleanSO _LongEQ(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongEQ:", so);  
    return BooleanSO.asBooleanSO(value == ((LongSO)so).value);
  }
  
 /**
  * Checks, if this long is greater or equal than its argument.
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected BooleanSO _LongGE(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongGE:", so);  
    return BooleanSO.asBooleanSO(value >= ((LongSO)so).value);
  }
      
 /**
  * Checks, if this long is greater than its argument.
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected BooleanSO _LongGT(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongGT:", so);  
    return BooleanSO.asBooleanSO(value > ((LongSO)so).value);
  }
  
 /**
  * Checks, if this long is less or equal than its argument.
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected BooleanSO _LongLE(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongLE:", so);  
    return BooleanSO.asBooleanSO(value <= ((LongSO)so).value);
  }
  
 /**
  * Checks, if this long is less than its argument.
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected BooleanSO _LongLT(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongLT:", so);  
    return BooleanSO.asBooleanSO(value < ((LongSO)so).value);
  }
  
 /**
  * Logical left shift of this long by the numbers of bits indicated
  * by the argument. 
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */  
  protected LongSO _LongLogicalShiftLeft(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongLogicalShiftLeft:", so);  
    return new LongSO(value << ((LongSO)so).value);
  }    

 /**
  * Logical right shift of this long by the numbers of bits indicated
  * by the argument. 0 is shifted into the most significant bit.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */  
  protected LongSO _LongLogicalShiftRight(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongLogicalShiftRight:", so);  
    return new LongSO(value >>> ((LongSO)so).value);
  }    
  
 /**
  * Returns this long modulo argument. 
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected LongSO _LongMod(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongMod:", so);  
    try{  
      return new LongSO(value % ((LongSO)so).value);  
    }
    catch(ArithmeticException e){
      throw new dSelfException(" Division by zero -> "+value+
         " _LongMod: 0 !", "_LongMod:", "divisionByZeroError");
    }  
  }  
  
 /**
  * Returns the product of this long and the argument. 
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected LongSO _LongMul(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongMul:", so);  
    return new LongSO(value * ((LongSO)so).value); 
  }  
  
 /**
  * Checks if this long is inequal to its receiver
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */ 
  protected BooleanSO _LongNE(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongNE:", so);  
    return BooleanSO.asBooleanSO(value != ((LongSO)so).value);
  }
  
 /**
  * Returns a bitwise OR of this long and the argument
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */ 
  protected LongSO _LongOr(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongOr:", so);  
    return new LongSO(value | ((LongSO)so).value);
  }  
  
 /**
  * Returns this long minus the argument.
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */
  protected LongSO _LongSub(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongSub:", so);  
    return new LongSO(value - ((LongSO)so).value); 
  }  
  
 /**
  * Returns a bitwise XOR of this long and the argument
  *
  * @param arg The argument type must be LongSO and is checked by
  * the method
  */ 
  protected LongSO _LongXor(dSelfObject so) throws dSelfException{

    checkIfLongSO("_LongXor:", so);  
    return new LongSO(value ^ ((LongSO)so).value);  
  }  
}

